#!/usr/bin/env bash
# Script to verify cosign signatures for chainver releases
# This verifies the checksums.txt signature, then uses it to verify individual artifacts

set -euo pipefail

ARTIFACT=${1:-}
if [ -z "$ARTIFACT" ]; then
    echo "Usage: $0 <artifact-file>"
    echo "Example: $0 chainver_1.2.3_Linux_x86_64.tar.gz"
    echo ""
    echo "This script verifies chainver artifacts using signed checksums."
    echo "Required files in the current directory:"
    echo "  - checksums.txt"
    echo "  - checksums.txt.sig"
    echo "  - checksums.txt.cert.pem"
    echo "  - <artifact-file> (optional, will verify if present)"
    exit 1
fi

# First, verify the checksums.txt signature
if [ ! -f "checksums.txt" ]; then
    echo "Error: checksums.txt not found"
    echo "Please download checksums.txt and its signature files from the release"
    exit 1
fi

if [ ! -f "checksums.txt.sig" ]; then
    echo "Error: checksums.txt.sig not found"
    exit 1
fi

if [ ! -f "checksums.txt.cert.pem" ]; then
    echo "Error: checksums.txt.cert.pem not found"
    exit 1
fi

echo "Verifying checksums.txt signature..."

# Verify the checksums.txt signature
cosign verify-blob \
    --certificate "checksums.txt.cert.pem" \
    --signature "checksums.txt.sig" \
    --certificate-identity-regexp "https://github.com/chainguard-dev/chainver/*" \
    --certificate-oidc-issuer "https://token.actions.githubusercontent.com" \
    "checksums.txt"

echo ""
echo "Certificate details:"
# Handle both PEM and base64-encoded certificates
if grep -q "BEGIN CERTIFICATE" checksums.txt.cert.pem 2>/dev/null; then
    openssl x509 -in "checksums.txt.cert.pem" -text -noout | grep -E "(Subject:|Issuer:|Not Before:|Not After:|URI:)" || true
else
    cat checksums.txt.cert.pem | base64 -d | openssl x509 -text -noout | grep -E "(Subject:|Issuer:|Not Before:|Not After:|URI:)" || true
fi

echo "✅ Checksums signature verified successfully!"

# Now verify the artifact checksum if an artifact was specified
if [ -n "$ARTIFACT" ] && [ -f "$ARTIFACT" ]; then
    echo ""
    echo "Verifying $ARTIFACT checksum..."
    if command -v sha256sum >/dev/null 2>&1; then
        if sha256sum -c --ignore-missing checksums.txt 2>/dev/null | grep -q "${ARTIFACT}: OK"; then
            echo "✅ Checksum verified successfully!"
        else
            echo "❌ Checksum verification failed!"
            exit 1
        fi
    else
        # macOS
        if shasum -a 256 -c checksums.txt 2>/dev/null | grep -q "${ARTIFACT}: OK"; then
            echo "✅ Checksum verified successfully!"
        else
            echo "❌ Checksum verification failed!"
            exit 1
        fi
    fi
else
    echo ""
    echo "No artifact specified or file not found. Skipping artifact verification."
    echo "To verify an artifact, ensure it's in the current directory and run:"
    echo "  $0 <artifact-filename>"
fi

echo ""
echo "Note: This verification confirms that:"
echo "  - The checksums file was signed by GitHub Actions"
echo "  - The signature is valid and matches the checksums"
echo "  - The signing certificate was issued by Fulcio"
if [ -n "$ARTIFACT" ] && [ -f "$ARTIFACT" ]; then
echo "  - The artifact matches its expected checksum"
fi